import curses as cur
from cell import Cell
import string

def test(scr):
    g = Grid(scr,6,8,0,0,8)
    scr.getch()
    g.heads()
    g.move(1,1)
    scr.getch()
    g.put_value_at(42,1,1)
    scr.addstr(cur.LINES-1,0,"Value 42 added in cell A1")
    scr.getch()

class BoundaryError(ValueError): pass

class Grid:
    ''' creates a grid of cells. 
    Cells are accessed using grid rather than window
    coordinates. Allows movement to a cell, and reversing 
    of display'''
    
    def __init__(self,win, ht, wd, y, x, cell_size=8):
        self.win = win
        self.headsOn = False
        self.selected = False
        self.select_start = None
        self.select_end = None
        self.size = (ht,wd)
        self.origin = (0,0)   # initial origin, heads off
        self.yx = self.origin # active cell
        self.cell_size = cell_size
        self.cells = []
        for r in range(ht):   # generate empty grid
            row_y = y + (r*3)
            row = [Cell(win,cell_size, row_y,col*(cell_size+2)+x)
                   for col in range(wd)]
            self.cells.append(row)

    def _map(self,y,x):   # helper function
        ''' maps grid coords to window coords'''
        wy = 1 + (3*y)
        wx = 1 + (self.cell_size+2) * x
        return wy,wx

    def _get_selection(self):
        if not self.selected:
            return []
        start_row,start_col = self.select_start
        end_row,end_col = self.select_end
        cells = []
        for row in range(start_row,end_row+1):
            for col in range(start_col,end_col+1):
                cells.append(self[row][col])
        return cells

    def move(self,y,x):
        ''' move cursor to cell y,x in grid'''
        if (y >= self.size[0] or  # check within boundaries
            x >= self.size[1] or
            y < self.origin[0] or 
            x < self.origin[1]):
            raise BoundaryError("%d or %d outside grid"%(y,x))
        ypt,xpt = self._map(y,x)
        self.win.move(ypt,xpt)   # uses curses move()
        self.yx = (y,x)

    def clear(self):
        for row in self[self.origin[0]:]:
            for cell in row[self.origin[1]:]:
                cell.value = None
                cell.refresh()
        self.move(*self.origin)

    def heads(self):
        ''' insert numbers along row 0 and letters down col 0
            reverses cells in row 0 and column 0
        '''
        letters = string.ascii_uppercase
        for num,cell in enumerate(self.cells[0]):
            if num > 0:
               cell.value= str(num).center(self.cell_size)
            cell.reverse()
        for index,row in enumerate(self.cells[1:]):
              row[0].value = letters[index]
              row[0].reverse()
        self.headsOn = True
        self.origin = (1,1)
        self.move(*self.origin)

    def cell_at(self, y,x):
        ''' Find cell with containing curses coords y,x.
            return grid y,x coordinates of cell '''
        row = y//3
        col = x//(self.cell_size+2)
        return row,col

    def make_selection(self,start,end):
        ''' show selected cells in inverse video and set flag attributes'''
        self.selected = True
        self.select_start = start
        self.select_end = end
        for row in range(start[0],end[0]+1):
            for col in range(start[1],end[1]+1):
                self[row][col].reverse()
        if start == end:  # need to reinvert first cell
            self[start[0]][start[1]].reverse()

    def deselect(self):
        ''' deselect the grid by reversing cells and resetting attributes'''
        for row in range(self.select_start[0],self.select_end[0]+1):
            for col in range(self.select_start[1],self.select_end[1]+1):
                self[row][col].reverse()
        self.selected = False
        self.select_start = None
        self.select_end = None
   
    def refresh(self):
        for row in self.cells:
            for cell in row:
                cell.refresh()


    # allow access to cell values using indexing
    def __setitem__(self,index,cell):
        ''' store value in cell specified by y,x grid coordinates'''
        self.cells[index] = cell

    def __getitem__(self,index):
        return self.cells[index]

class SummingGrid(Grid):
    '''Adds ability to total a column to basic Grid'''

    def __init__(self,win,ht,wd,y,x, cell_size=8):
        super().__init__(win,ht,wd,y,x,cell_size)

    def sum(self,col):
        ''' sums all values in current column'''
        first = 1 if self.headsOn else 0
        if self.selected:
            cells = self._get_selection()
            vals = [cell.value for cell in cells if cell.value]
        else:
            vals = [row[col].value for row in self[first:] if row[col].value]
        return sum(vals)
    
if __name__ == "__main__": cur.wrapper(test)
