import curses as cur
from grid import SummingGrid, BoundaryError

# define mouse event indices
X_COORD = 1
Y_COORD = 2
BUTTON_STATE = 4

class Totalizer:
    ''' spreadsheet like grid that can display totals of columns.
        values must be integers.'''
        
    help_string = "Press F1(or H) for help, X to eXit."

    def __init__(self, win, ht,wd, cell_size=8):
        self.win = win
        self.size = (ht,wd)
        self.cell_size = cell_size
        self.selecting = False
        self.selection = None
        self.grid = SummingGrid(win,ht,wd, 0,0, cell_size)
        self.grid.heads()
        cur.mousemask(cur.ALL_MOUSE_EVENTS)

    def _show_status(self,msg):
        ''' display message on bottom line of window '''
        height,width = self.win.getmaxyx()
        y,x = self.grid.yx
        self.win.addstr(height-2,1, ' ' * (width-2))  #clear line but not border
        self.win.addstr(height-2,1, msg)
        self.grid.move(y,x)

    def _show_help(self):
        ''' show help screen in new window, remove after any keypress'''
        ht,wd = self.win.getmaxyx()
        top = (ht-15)//2
        left = (wd-50)//2 
        win = cur.newwin(15,50,top,left)
        win.keypad(True)
        win.box(0,0)
        win.addstr(1,2,"Arrow keys move cursor")
        win.addstr(3,2,"Mouse positions cursor")
        win.addstr(5,2,"=N<ret> inserts value")
        win.addstr(7,2,"+ sums the current column or selection")
        win.addstr(9,2,"S start/end/cancel selection")
        win.addstr(11,2,"C clear grid")
        win.addstr(13,2,"X eXit the application")
        win.refresh()
        win.getch()
        win.erase()
        self.grid.refresh()
        self.grid.move(*self.grid.yx) # restore cursor to previous cell

    def run(self):
        ''' start the event loop, process the actions '''
        Y,X = self.win.getmaxyx()
        self.win.addstr(Y-3,1, self.help_string)
        self.grid.move(1,1)  # starting position
        while True:
           key = self.win.getch()
           if key in [ ord('X'),ord('x')] : break
           if key in [cur.KEY_F1, ord('h'), ord('H')]:
               self._show_help()
           if key in [cur.KEY_UP, ord('k')]:
               y,x = self.grid.yx
               try: self.grid.move(y-1,x)
               except BoundaryError as e:
                  self._show_status(e.args[0])
           elif key in [cur.KEY_DOWN, ord('j')]:
               y,x = self.grid.yx
               try: self.grid.move(y+1,x)
               except BoundaryError as e:
                  self._show_status(e.args[0])
           elif key in [ cur.KEY_LEFT, ord('h')]:
               y,x = self.grid.yx 
               try: self.grid.move(y,x-1)
               except BoundaryError as e:
                  self._show_status(e.args[0])
           elif key in [ cur.KEY_RIGHT, ord('l')]:
               y,x = self.grid.yx
               try: self.grid.move(y,x+1)
               except BoundaryError as e:
                  self._show_status(e.args[0])
           elif key == ord('='):
               val = self.win.getstr()
               y,x = self.grid.yx
               try:
                  self.grid[y][x].value = int(val)  #only want integers
                  self.grid.move(y+1,x)
               except BoundaryError: pass   # leave it as is
               except ValueError:  
                  self._show_status("Error: invalid value - %s" % val)
           elif key == ord('+'):
               y,x = self.grid.yx
               tot = self.grid.sum(x)
               if self.selection: 
                   self._show_status("Selection total = %d" % tot)
               else: 
                   self._show_status("Column %d total = %d" % (x,tot))
           elif key == ord('S'):
               if not self.selecting and not self.selection:   # first S
                   self.selecting = True
                   self.selection = self.grid.yx
                   self.grid.make_selection(self.selection,self.selection) # mark start
                   self._show_status("Selection ON")
               elif self.selecting:     # second S
                   self.selecting = False
                   self.grid.make_selection(self.selection, self.grid.yx)
                   self._show_status("Selection complete")
               else:                    # third S
                   self.grid.deselect()
                   self.selection = None
                   self._show_status("Selection off")
           elif key == ord("C"):
               self.grid.clear()
           elif key == cur.KEY_MOUSE:
               m_event = cur.getmouse()
               if m_event[BUTTON_STATE] | cur.BUTTON1_CLICKED:
                   my = m_event[Y_COORD]
                   mx = m_event[X_COORD]
                   gy,gx = self.grid.cell_at(my,mx)
                   try: self.grid.move(gy,gx)
                   except BoundaryError: pass  # leave it as-is
        
def main(scr):
    cur.raw()
    scr.box(0,0)
    scr.refresh()
    rows,cols = scr.getmaxyx()
    grid_rows = (rows-4)//3
    grid_cols = cols//8
    totalizer = Totalizer(scr,grid_rows,grid_cols,6)
    totalizer.run()

if __name__ == "__main__": cur.wrapper(main)
